<?php
/**
 * List Users API (Member Directory)
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = authenticate();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendResponse('error', 'Method not allowed', null, 405);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = USERS_PER_PAGE;
    $offset = ($page - 1) * $limit;
    $search = isset($_GET['search']) ? sanitizeInput($_GET['search']) : '';

    $where_clause = "WHERE u.is_active = 1";
    if ($search) {
        $where_clause .= " AND (u.username LIKE :search OR u.full_name LIKE :search OR u.location LIKE :search)";
    }

    $query = "SELECT 
                u.id,
                u.username,
                u.full_name,
                u.profile_picture,
                u.location,
                u.supporter_id,
                u.role,
                u.created_at,
                (SELECT last_activity FROM online_users WHERE user_id = u.id) as last_activity,
                (SELECT COUNT(*) FROM messages WHERE user_id = u.id AND is_deleted = 0) as message_count
              FROM users u
              $where_clause
              ORDER BY u.created_at DESC
              LIMIT :limit OFFSET :offset";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
    
    if ($search) {
        $search_param = "%$search%";
        $stmt->bindParam(':search', $search_param);
    }
    
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Check online status
    foreach ($users as &$u) {
        $u['is_online'] = false;
        if ($u['last_activity']) {
            $last_activity = strtotime($u['last_activity']);
            $u['is_online'] = (time() - $last_activity) < 300;
        }
        unset($u['last_activity']);
    }

    // Get total count
    $count_query = "SELECT COUNT(*) as total FROM users u $where_clause";
    $count_stmt = $conn->prepare($count_query);
    if ($search) {
        $count_stmt->bindParam(':search', $search_param);
    }
    $count_stmt->execute();
    $total = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];

    sendResponse('success', 'Users retrieved', [
        'users' => $users,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ], 200);

} catch (PDOException $e) {
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
}
?>
