<?php
// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

/**
 * User Registration API
 */

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse('error', 'Method not allowed', null, 405);
}

$data = json_decode(file_get_contents("php://input"));

// Validate required fields
if (empty($data->username) || empty($data->email) || empty($data->password) || empty($data->full_name)) {
    sendResponse('error', 'All fields are required', null, 400);
}

// Validate email format
if (!filter_var($data->email, FILTER_VALIDATE_EMAIL)) {
    sendResponse('error', 'Invalid email format', null, 400);
}

// Validate password strength (min 8 chars, 1 uppercase, 1 lowercase, 1 number)
if (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/', $data->password)) {
    sendResponse('error', 'Password must be at least 8 characters with uppercase, lowercase, and number', null, 400);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    // Check if username or email already exists
    $check_query = "SELECT id FROM users WHERE username = :username OR email = :email";
    $check_stmt = $conn->prepare($check_query);
    $check_stmt->bindParam(':username', $data->username);
    $check_stmt->bindParam(':email', $data->email);
    $check_stmt->execute();

    if ($check_stmt->rowCount() > 0) {
        sendResponse('error', 'Username or email already exists', null, 409);
    }

    // Generate supporter ID
    $supporter_id = 'CS2028-' . strtoupper(substr(uniqid(), -8));

    // Hash password
    $password_hash = password_hash($data->password, PASSWORD_BCRYPT);

    // Generate verification token
    $verification_token = generateToken();

    // Insert new user
    $insert_query = "INSERT INTO users 
        (username, email, password_hash, full_name, phone, location, supporter_id, verification_token) 
        VALUES 
        (:username, :email, :password_hash, :full_name, :phone, :location, :supporter_id, :verification_token)";
    
    $insert_stmt = $conn->prepare($insert_query);
    $insert_stmt->bindParam(':username', sanitizeInput($data->username));
    $insert_stmt->bindParam(':email', sanitizeInput($data->email));
    $insert_stmt->bindParam(':password_hash', $password_hash);
    $insert_stmt->bindParam(':full_name', sanitizeInput($data->full_name));
    $insert_stmt->bindParam(':phone', sanitizeInput($data->phone ?? ''));
    $insert_stmt->bindParam(':location', sanitizeInput($data->location ?? ''));
    $insert_stmt->bindParam(':supporter_id', $supporter_id);
    $insert_stmt->bindParam(':verification_token', $verification_token);

    if ($insert_stmt->execute()) {
        $user_id = $conn->lastInsertId();

        // Log activity
        $log_query = "INSERT INTO activity_logs (user_id, activity_type, description, ip_address) 
                      VALUES (:user_id, 'registration', 'New user registered', :ip)";
        $log_stmt = $conn->prepare($log_query);
        $log_stmt->bindParam(':user_id', $user_id);
        $log_stmt->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
        $log_stmt->execute();

        sendResponse('success', 'Registration successful', [
            'user_id' => $user_id,
            'supporter_id' => $supporter_id,
            'verification_token' => $verification_token
        ], 201);
       } else {
        $errorInfo = $insert_stmt->errorInfo();
        sendResponse('error', 'Registration failed: ' . ($errorInfo[2] ?? 'Unknown database error'), null, 500);
    }

} catch (PDOException $e) {
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    sendResponse('error', 'Server error: ' . $e->getMessage(), null, 500);
}
?>