<?php
/**
 * Admin User Management API
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = requireAdmin();

if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    // Update user (ban, change role, etc.)
    $data = json_decode(file_get_contents("php://input"));

    if (empty($data->user_id)) {
        sendResponse('error', 'User ID is required', null, 400);
    }

    try {
        $database = new Database();
        $conn = $database->getConnection();

        $updates = [];
        $params = [':user_id' => $data->user_id];

        if (isset($data->is_active)) {
            $updates[] = "is_active = :is_active";
            $params[':is_active'] = (bool)$data->is_active;
        }

        if (isset($data->role)) {
            if (!in_array($data->role, ['member', 'moderator', 'admin'])) {
                sendResponse('error', 'Invalid role', null, 400);
            }
            $updates[] = "role = :role";
            $params[':role'] = $data->role;
        }

        if (empty($updates)) {
            sendResponse('error', 'No fields to update', null, 400);
        }

        $query = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = :user_id";
        $stmt = $conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }

        if ($stmt->execute()) {
            // Log activity
            $action = isset($data->is_active) && !$data->is_active ? 'banned' : 'updated';
            $log_query = "INSERT INTO activity_logs (user_id, activity_type, description, ip_address) 
                          VALUES (:admin_id, 'admin_action', :description, :ip)";
            $log_stmt = $conn->prepare($log_query);
            $description = "Admin {$action} user ID: {$data->user_id}";
            $log_stmt->bindParam(':admin_id', $user['id']);
            $log_stmt->bindParam(':description', $description);
            $log_stmt->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
            $log_stmt->execute();

            sendResponse('success', 'User updated successfully', null, 200);
        } else {
            sendResponse('error', 'Failed to update user', null, 500);
        }

    } catch (PDOException $e) {
        sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
    }

} elseif ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    // Delete user permanently
    $data = json_decode(file_get_contents("php://input"));

    if (empty($data->user_id)) {
        sendResponse('error', 'User ID is required', null, 400);
    }

    try {
        $database = new Database();
        $conn = $database->getConnection();

        $query = "DELETE FROM users WHERE id = :user_id";
        $stmt = $conn->prepare($query);
        $stmt->bindParam(':user_id', $data->user_id);

        if ($stmt->execute()) {
            // Log activity
            $log_query = "INSERT INTO activity_logs (user_id, activity_type, description, ip_address) 
                          VALUES (:admin_id, 'admin_action', :description, :ip)";
            $log_stmt = $conn->prepare($log_query);
            $description = "Admin deleted user ID: {$data->user_id}";
            $log_stmt->bindParam(':admin_id', $user['id']);
            $log_stmt->bindParam(':description', $description);
            $log_stmt->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
            $log_stmt->execute();

            sendResponse('success', 'User deleted successfully', null, 200);
        } else {
            sendResponse('error', 'Failed to delete user', null, 500);
        }

    } catch (PDOException $e) {
        sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
    }

} else {
    sendResponse('error', 'Method not allowed', null, 405);
}
?>
