<?php
/**
 * Admin Dashboard Statistics API
 */

// Define the root path of the project
define('ROOT_PATH', realpath(__DIR__ . '/../../..'));

require_once ROOT_PATH . '/backend/api/cors.php';
handleCORS();

require_once ROOT_PATH . '/backend/config/database.php';
require_once ROOT_PATH . '/backend/config/config.php';
require_once ROOT_PATH . '/backend/middleware/auth.php';

$user = requireAdmin();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendResponse('error', 'Method not allowed', null, 405);
}

try {
    $database = new Database();
    $conn = $database->getConnection();

    // Total users
    $users_query = "SELECT COUNT(*) as total, 
                           SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
                           SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today
                    FROM users";
    $users_stmt = $conn->query($users_query);
    $users_stats = $users_stmt->fetch(PDO::FETCH_ASSOC);

    // Total messages
    $messages_query = "SELECT COUNT(*) as total,
                              SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                              SUM(CASE WHEN DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as week
                       FROM messages WHERE is_deleted = 0";
    $messages_stmt = $conn->query($messages_query);
    $messages_stats = $messages_stmt->fetch(PDO::FETCH_ASSOC);

    // Online users
    $online_query = "SELECT COUNT(*) as count 
                     FROM online_users 
                     WHERE last_activity > DATE_SUB(NOW(), INTERVAL 5 MINUTE)";
    $online_stmt = $conn->query($online_query);
    $online_count = $online_stmt->fetch(PDO::FETCH_ASSOC)['count'];

    // Announcements
    $announcements_query = "SELECT COUNT(*) as total FROM announcements";
    $announcements_stmt = $conn->query($announcements_query);
    $announcements_count = $announcements_stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Most active users (top 10)
    $active_users_query = "SELECT 
                            u.id,
                            u.username,
                            u.full_name,
                            u.profile_picture,
                            COUNT(m.id) as message_count
                          FROM users u
                          LEFT JOIN messages m ON u.id = m.user_id AND m.is_deleted = 0
                          WHERE u.is_active = 1
                          GROUP BY u.id
                          ORDER BY message_count DESC
                          LIMIT 10";
    $active_users_stmt = $conn->query($active_users_query);
    $active_users = $active_users_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Recent activity (handle if table doesn't exist)
    $recent_activity = [];
    try {
        $activity_query = "SELECT 
                            a.activity_type,
                            a.description,
                            a.created_at,
                            u.username,
                            u.full_name
                          FROM activity_logs a
                          LEFT JOIN users u ON a.user_id = u.id
                          ORDER BY a.created_at DESC
                          LIMIT 20";
        $activity_stmt = $conn->query($activity_query);
        $recent_activity = $activity_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table might not exist, use empty array
        $recent_activity = [];
    }

    // User growth (last 7 days)
    $growth_query = "SELECT 
                        DATE(created_at) as date,
                        COUNT(*) as count
                     FROM users
                     WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                     GROUP BY DATE(created_at)
                     ORDER BY date ASC";
    $growth_stmt = $conn->query($growth_query);
    $user_growth = $growth_stmt->fetchAll(PDO::FETCH_ASSOC);

    sendResponse('success', 'Dashboard stats retrieved', [
        'users' => [
            'total' => (int)$users_stats['total'],
            'active' => (int)$users_stats['active'],
            'today' => (int)$users_stats['today'],
            'online' => (int)$online_count
        ],
        'messages' => [
            'total' => (int)$messages_stats['total'],
            'today' => (int)$messages_stats['today'],
            'week' => (int)$messages_stats['week']
        ],
        'announcements' => [
            'total' => (int)$announcements_count
        ],
        'active_users' => $active_users,
        'recent_activity' => $recent_activity,
        'user_growth' => $user_growth
    ], 200);

} catch (PDOException $e) {
    error_log('Admin Dashboard Error: ' . $e->getMessage());
    sendResponse('error', 'Database error: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log('Admin Dashboard Error: ' . $e->getMessage());
    sendResponse('error', 'Server error: ' . $e->getMessage(), null, 500);
}
?>
